function varargout = cbricks(varargin)
%CBRICKS
% Generates MATLAB data structures to visualize a set of curved bricks generated
% by the MAGPIE code.
%
%     FVB = cbricks(BDATA) returns a face-vertex structure that can be used to
%     visualize the bricks. BDATA is a 6-column array imported from a 
%     '_cbricks.txt' file generated by MAGPIE for an array of cbrick magnets.
%     The bricks may be visualized via a call to the patch function; 
%     for example:
%         patch(FVB, 'EdgeColor', 'None');
%
%     FVB = cbricks(R0, R1, Z0, Z1, PHI0, PHI1) also returns the face-vertex
%     structure for visualization, but the inputs are individual arrays of
%     coordinate bounds for each brick rather than a single array.
%     R0 and R1 are the inner and outer radial bounds for each brick, 
%     respectively; Z0 and Z1 are the lower and upper vertical bounds; 
%     PHI0 and PHI1 are the lower and upper bounds in azimuthal angle in 
%     radians.
%
%     [FVB, FVF] = cbricks(...) returns an additional face-vertex structure
%     (FVF) defining a wireframe for each brick. The wireframes may be 
%     visualized via a call to patch(); for example:
%         patch(FVF, 'FaceColor', 'None');
%
%     FVB = cbricks(..., MAGNITUDE) returns a face-vertex structure with 
%     'FaceVertexCData' values assigned to each brick according to the elements
%     in MAGNITUDE, an array containing one scalar value for each brick.
%
% Author:  K. C. Hammond
% Contact: khammond@pppl.gov
%     
 
 
    %---- Input checking -------------------------------------------------------

    % Output arguments
    if (nargout ~= 1 && nargout ~= 2)
        error('Must have one or two output arguments');
    end

    % Input arguments
    if nargin < 6
        if (ndims(varargin{1}) ~= 2 && size(varargin{1}) ~= 6)
            error(['Input array BDATA must be a 2D array with 6 columns or ' ...
                   'supplied as individual arrays with r, z, and phi bounds']);
        else
            r0 = varargin{1}(:,1);
            r1 = varargin{1}(:,2);
            z0 = varargin{1}(:,3);
            z1 = varargin{1}(:,4);
            phi0 = varargin{1}(:,5);
            phi1 = varargin{1}(:,6);
        end
        if nargin == 2
            magnitude = varargin{2};
        end
        if nargin > 2
            error('Incorrect number of input arguments');
        end
    elseif (nargin == 6 || nargin == 7)
        [r0, r1, z0, z1, phi0, phi1] = varargin{1:6};
        if nargin == 7
            magnitude = varargin{7};
        end
    else
        error('Incorrect number of input arguments');
    end

    % Size of the input variables
    nBricks = numel(r0);
    if (numel(r1) ~= nBricks ...
           || numel(z0) ~= nBricks ...
           || numel(z1) ~= nBricks ...
           || numel(phi0) ~= nBricks ...
           || numel(phi1) ~= nBricks ...
           || (exist('magnitude', 'var') && numel(magnitude) ~= nBricks))
        error('All input arrays must have the same number of elements');
    end

    %---- Coordinates of points along the brick edges --------------------------

    % Azimuthal angles for points along the curved edges of the bricks
    nPhi = round(max(phi1(:) - phi0(:))*360/(2*pi));  % at least one pt per deg
    dPhi = (phi1(:) - phi0(:))./(nPhi - 1);
    phi = repmat(phi0(:)', nPhi, 1) + ...
              repmat(dPhi', nPhi, 1).*repmat([0:(nPhi-1)]', 1, nBricks);
    
    % Cartesian coordinates of the four curved edges of each brick
    xIn  = repmat(r0(:)', nPhi, 1).*cos(phi);
    xOut = repmat(r1(:)', nPhi, 1).*cos(phi);
    yIn  = repmat(r0(:)', nPhi, 1).*sin(phi);
    yOut = repmat(r1(:)', nPhi, 1).*sin(phi);
    zLo  = repmat(z0(:)', nPhi, 1);
    zUp  = repmat(z1(:)', nPhi, 1);
   
    % List of all "vertices" (i.e., points on edges) for the set of bricks
    fvb.vertices = [[xIn(:); xOut(:); xOut(:); xIn(:)] ...
                   [yIn(:); yOut(:); yOut(:); yIn(:)] ...
                   [zLo(:); zLo(:);  zUp(:);  zUp(:)]];
 
    %---- Definition of the facets for 3d brick representation -----------------

    % Row indices of the vertex coordinates as they appear in fvb.vertices
    indMatrix = reshape(1:(nPhi*nBricks), nPhi, nBricks);
    indInLo   = indMatrix + 0*nPhi*nBricks;
    indOutLo  = indMatrix + 1*nPhi*nBricks;
    indOutUp  = indMatrix + 2*nPhi*nBricks;
    indInUp   = indMatrix + 3*nPhi*nBricks;
 
    % Define quadrilateral facets along each curved and annular face
    facesPerBrick = 4*(nPhi-1) + 2;
    fvb.faces = zeros(nBricks*facesPerBrick, 4);
    for i = 1:(nPhi-1)
        iStart = (i-1)*nBricks*4;
        rangeInner = (1:nBricks) + iStart;
        rangeUpper = (1:nBricks) + iStart + nBricks;
        rangeOuter = (1:nBricks) + iStart + 2*nBricks;
        rangeLower = (1:nBricks) + iStart + 3*nBricks;
        fvb.faces(rangeInner,:) = ...
            [indInUp(i,:)' indInUp(i+1,:)' indInLo(i+1,:)' indInLo(i,:)'];
        fvb.faces(rangeUpper,:) = ...
            [indInUp(i,:)' indInUp(i+1,:)' indOutUp(i+1,:)' indOutUp(i,:)'];
        fvb.faces(rangeOuter,:) = ...
            [indOutUp(i,:)' indOutUp(i+1,:)' indOutLo(i+1,:)' indOutLo(i,:)'];
        fvb.faces(rangeLower,:) = ...
            [indOutLo(i,:)' indOutLo(i+1,:)' indInLo(i+1,:)' indInLo(i,:)'];
    end    

    % Define facets for the front and back faces (constant azimuthal angle)
    rangeBack  = (1:nBricks) + nBricks*(facesPerBrick-2);
    rangeFront = (1:nBricks) + rangeBack(end);
    fvb.faces(rangeBack,:) = ...
        [indInLo(1,:)'    indOutLo(1,:)'    indOutUp(1,:)'    indInUp(1,:)'];
    fvb.faces(rangeFront,:) = ...
        [indInLo(nPhi,:)' indOutLo(nPhi,:)' indOutUp(nPhi,:)' indInUp(nPhi,:)'];

    % Assign a magnitude to each brick (and all its facets) if provided
    if exist('magnitude', 'var')
        fvb.FaceVertexCData = repmat(magnitude(:), facesPerBrick, 1);
    end

    %---- Wireframe for each of the bricks -------------------------------------

    fvf.vertices = fvb.vertices;

    framesPerBrick = 4;
    indFrameBot   = (1:nBricks);
    indFrameTop   = (1:nBricks) + 1*nBricks;
    indFrameBack  = (1:nBricks) + 2*nBricks;
    indFrameFront = (1:nBricks) + 3*nBricks;
    fvf.faces = nan(nBricks*framesPerBrick, max(4, 2*nPhi));
    fvf.faces(indFrameBot,1:2*nPhi) = [indInLo' fliplr(indOutLo')];
    fvf.faces(indFrameTop,1:2*nPhi) = [indInUp' fliplr(indOutUp')];
    fvf.faces(indFrameBack,1:4) = fvb.faces(rangeBack,:);
    fvf.faces(indFrameFront,1:4) = fvb.faces(rangeFront,:);

    %---- Output parameters ----------------------------------------------------

    varargout{1} = fvb;
    if nargout == 2
        varargout{2} = fvf;
    end

end 
 
