function [fvstruct]  = qhex(vertex_data, magnitude)
%QHEX(vertex_data, magnitude)
% Generates a patch structure for a set of vertex data for quadrilaterally-
% faced hexahedra
%
% Parameters
% ----------
%    vertex_data: double array
%        Array with the coordinates of the vertices of each hexahedron. Each
%        row corresponds to a different trapezoid. Within each row, the columns
%        are ordered as follows: x1, y1, z1, x2, y2, z2, ..., x8, y8, z8.
%        Here, the number labels the vertex, and x, y, and z are the 
%        Cartesian x, y, and z components of the respective vertex.
%        Vertices 1-4 are assumed to go around the perimeter of one of the
%        parallel faces; vertices 5-8 are assumed to go around the perimeter
%        of the other parallel face in the same order such that the pairs
%        1-5, 2-6, 3-7, and 4-8 are connected by edges.
%    magnitude: double array
%        (Optional) values corresponding to each solid. If provided, 
%        the FaceVertexCData field will be included so that the patch
%        structure will color each trapezoidal solid according to its 
%        respective magnetude.
%
% Returns
% -------
%    fvstruct: face-vertex structure
%        Structure that can be used for generating a patch representation of
%        the prisms.
%
% Author:  K. C. Hammond
% Contact: khammond@pppl.gov
%

    % x coordinates of each of 8 vertices of each prism
    v1x = vertex_data(:,1);
    v2x = vertex_data(:,4);
    v3x = vertex_data(:,7);
    v4x = vertex_data(:,10);
    v5x = vertex_data(:,13);
    v6x = vertex_data(:,16);
    v7x = vertex_data(:,19);
    v8x = vertex_data(:,22);

    % y coordinates of each of 8 vertices of each prism
    v1y = vertex_data(:,2);
    v2y = vertex_data(:,5);
    v3y = vertex_data(:,8);
    v4y = vertex_data(:,11);
    v5y = vertex_data(:,14);
    v6y = vertex_data(:,17);
    v7y = vertex_data(:,20);
    v8y = vertex_data(:,23);

    % z coordinates of each of 8 vertices of each prism
    v1z = vertex_data(:,3);
    v2z = vertex_data(:,6);
    v3z = vertex_data(:,9);
    v4z = vertex_data(:,12);
    v5z = vertex_data(:,15);
    v6z = vertex_data(:,18);
    v7z = vertex_data(:,21);
    v8z = vertex_data(:,24);

    % Vertex array
    fvstruct.vertices = [v1x v1y v1z; ...
                         v2x v2y v2z; ...
                         v3x v3y v3z; ...
                         v4x v4y v4z; ...
                         v5x v5y v5z; ...
                         v6x v6y v6z; ...
                         v7x v7y v7z; ...
                         v8x v8y v8z ];

    % Assign indices for each vertex
    nTraps = length(v1x);
    i1 = (1:nTraps)';
    i2 = i1 + nTraps;
    i3 = i2 + nTraps;
    i4 = i3 + nTraps;
    i5 = i4 + nTraps;
    i6 = i5 + nTraps;
    i7 = i6 + nTraps;
    i8 = i7 + nTraps;

    % Matrix of indices for each face
    fvstruct.faces = [i1 i2 i3 i4; ...
                      i5 i6 i7 i8; ...
                      i1 i2 i6 i5; ...
                      i2 i3 i7 i6; ...
                      i3 i4 i8 i7; ...
                      i4 i1 i5 i8 ];

    % CData (if magnitude is given)
    if exist('magnitude', 'var')
        fvstruct.FaceVertexCData = repmat(magnitude(:), 6, 1);
    end

end

